#!/bin/bash

# Function to find the gemini executable
find_gemini() {
    set +e  # Disable exit on error within this function

    # First, try to load user's shell environment to get proper PATH
    # This is crucial when running from GUI apps which have minimal PATH
    # Use subshells to prevent returns/exits in profile files from affecting this function
    if [ -f "$HOME/.zshrc" ]; then
        (source "$HOME/.zshrc") &> /dev/null || true
        # Re-export PATH from a new shell that sources the profile
        eval "$(bash -c 'source "$HOME/.zshrc" 2>/dev/null && echo "export PATH=\"$PATH\""' 2>/dev/null || true)"
    elif [ -f "$HOME/.bashrc" ]; then
        (source "$HOME/.bashrc") &> /dev/null || true
        eval "$(bash -c 'source "$HOME/.bashrc" 2>/dev/null && echo "export PATH=\"$PATH\""' 2>/dev/null || true)"
    elif [ -f "$HOME/.bash_profile" ]; then
        (source "$HOME/.bash_profile") &> /dev/null || true
        eval "$(bash -c 'source "$HOME/.bash_profile" 2>/dev/null && echo "export PATH=\"$PATH\""' 2>/dev/null || true)"
    fi

    # Also try to load profile which is more commonly used on macOS
    if [ -f "$HOME/.profile" ]; then
        (source "$HOME/.profile") &> /dev/null || true
        eval "$(bash -c 'source "$HOME/.profile" 2>/dev/null && echo "export PATH=\"$PATH\""' 2>/dev/null || true)"
    fi

    # Check if gemini is already in PATH (after loading shell environment)
    if command -v gemini &> /dev/null; then
        command -v gemini
        return 0
    fi

    # Try to source nvm and find gemini
    if [ -f "$HOME/.nvm/nvm.sh" ]; then
        export NVM_DIR="$HOME/.nvm"
        (source "$NVM_DIR/nvm.sh") &> /dev/null || true
        eval "$(bash -c 'export NVM_DIR="$HOME/.nvm"; source "$NVM_DIR/nvm.sh" 2>/dev/null && echo "export PATH=\"$PATH\""' 2>/dev/null || true)"
        if command -v gemini &> /dev/null; then
            command -v gemini
            return 0
        fi
    fi

    # Try to source asdf if installed
    if [ -f "$HOME/.asdf/asdf.sh" ]; then
        (source "$HOME/.asdf/asdf.sh") &> /dev/null || true
        eval "$(bash -c 'source "$HOME/.asdf/asdf.sh" 2>/dev/null && echo "export PATH=\"$PATH\""' 2>/dev/null || true)"
        if command -v gemini &> /dev/null; then
            command -v gemini
            return 0
        fi
    fi

    # Check for Homebrew and add to PATH if found
    if [ -x "/opt/homebrew/bin/brew" ]; then
        eval "$(/opt/homebrew/bin/brew shellenv 2>/dev/null || true)"
    elif [ -x "/usr/local/bin/brew" ]; then
        eval "$(/usr/local/bin/brew shellenv 2>/dev/null || true)"
    fi

    # Check again after setting up Homebrew
    if command -v gemini &> /dev/null; then
        command -v gemini
        return 0
    fi

    # Check common npm global installation directories and Homebrew
    local common_paths=(
        # Homebrew installation
        "/opt/homebrew/bin/gemini"
        "/usr/local/bin/gemini"
        # NVM installations
        "$HOME/.nvm/versions/node/*/bin/gemini"
        # Homebrew node installations
        "/opt/homebrew/lib/node_modules/.bin/gemini"
        "/usr/local/lib/node_modules/.bin/gemini"
        # npm global
        "$HOME/.npm-global/bin/gemini"
        "$HOME/.npm/bin/gemini"
        # Yarn global
        "$HOME/.config/yarn/global/node_modules/.bin/gemini"
        "$HOME/.yarn/bin/gemini"
        # pnpm global
        "$HOME/.local/share/pnpm/gemini"
        "$HOME/Library/pnpm/gemini"
        # asdf
        "$HOME/.asdf/shims/gemini"
        # fnm
        "$HOME/.fnm/node-versions/*/installation/bin/gemini"
        "$HOME/Library/Application Support/fnm/node-versions/*/installation/bin/gemini"
        # volta
        "$HOME/.volta/bin/gemini"
        # macOS user local
        "$HOME/.local/bin/gemini"
        "/usr/bin/gemini"
    )

    for path_pattern in "${common_paths[@]}"; do
        # Use ls to expand glob patterns, pick the most recent match
        local expanded_path=$(ls -t $path_pattern 2>/dev/null | head -n 1)
        if [ -n "$expanded_path" ] && [ -x "$expanded_path" ]; then
            echo "$expanded_path"
            return 0
        fi
    done

    set -e  # Re-enable exit on error
    return 1
}

# Enable exit on error for main script
set -e

# Find gemini command
GEMINI_CMD=$(find_gemini) || true
if [ -z "$GEMINI_CMD" ]; then
    echo "[error] Google Gemini CLI not found." >&2
    echo "[error] " >&2
    echo "[error] Please install it with one of:" >&2
    echo "[error]   npm install -g @google/gemini-cli" >&2
    echo "[error]   brew install gemini-cli" >&2
    echo "[error] " >&2
    echo "[error] Or run without installing:" >&2
    echo "[error]   npx @google/gemini-cli" >&2
    echo "[error] " >&2
    echo "[error] Current PATH: $PATH" >&2
    echo "[error] HOME: $HOME" >&2
    echo "[error] " >&2
    echo "[error] If installed, ensure 'gemini' is in your PATH by adding it to ~/.zshrc or ~/.bashrc" >&2
    exit 1
fi

# Read inputs from environment variables
REPO_PATH="$TOWER_EXTENSION_INPUT_REPOSITORYPATH"
USER_PROMPT="$TOWER_EXTENSION_INPUT_USERPROMPT"
PROMPT_INPUT="$TOWER_EXTENSION_INPUT_PROMPT"
CURRENT_MESSAGE="$TOWER_EXTENSION_INPUT_CURRENTMESSAGE"
FUNCTION_NAME="$TOWER_EXTENSION_FUNCTION"

# Determine which function to execute
if [ "$FUNCTION_NAME" = "rephraseCommitMessage" ]; then
    # REPHRASE MODE

    # Check if we have a current message
    if [ -z "$CURRENT_MESSAGE" ]; then
        echo "[error] No current message provided for rephrasing" >&2
        exit 1
    fi

    # Report progress
    echo "[progress] Analyzing commit message..." >&2

    FUNCTION_PROMPT="Rephrase the commit message per the user prompt. Only output the commit message string, no surrounding markdown or explanations."
    PROMPT="$FUNCTION_PROMPT

Context:
repositoryPath: $REPO_PATH
currentMessage:
\`\`\`
$CURRENT_MESSAGE
\`\`\`

User prompt:
$USER_PROMPT"

    # Report progress
    echo "[progress] Rephrasing commit message..." >&2

    if [ -n "$REPO_PATH" ]; then
        cd "$REPO_PATH"
    fi

    # Call Gemini - it should only output the rephrased commit message
    # Use --yolo mode to auto-approve all actions and avoid any interactive prompts
    COMMIT_MESSAGE=$("$GEMINI_CMD" --yolo "$PROMPT" 2>/dev/null)

elif [ "$FUNCTION_NAME" = "generateFromPrompt" ]; then
    # GENERATE FROM PROMPT MODE

    # Check if we have a user prompt
    if [ -z "$USER_PROMPT" ]; then
        echo "[error] No prompt provided" >&2
        exit 1
    fi

    # Report progress
    echo "[progress] Generating commit message from prompt..." >&2

    DRAFT_MESSAGE="$CURRENT_MESSAGE"

    FUNCTION_PROMPT="Create a commit message based on the user prompt and the provided draft. Only access the repository if explicitly requested by the user. Only output the commit message string, no surrounding markdown or explanations."
    PROMPT="$FUNCTION_PROMPT

Context:
repositoryPath: $REPO_PATH
currentMessage:
\`\`\`
$CURRENT_MESSAGE
\`\`\`

User prompt:
$USER_PROMPT

Draft message:
\`\`\`
$DRAFT_MESSAGE
\`\`\`"

    if [ -n "$REPO_PATH" ]; then
        cd "$REPO_PATH"
    fi

    # Call Gemini with just the prompt - it should only output the commit message
    # Use --yolo mode to auto-approve all actions and avoid any interactive prompts
    COMMIT_MESSAGE=$("$GEMINI_CMD" --yolo "$PROMPT" 2>/dev/null)

else
    # GENERATE MODE (default)

    # Report progress
    echo "[progress] Reading repository changes..." >&2

    # Report progress
    echo "[progress] Analyzing changes with Google Gemini..." >&2

    FUNCTION_PROMPT="Generate a commit message based on the diff of staged changes. Follow the user prompt. Only output the commit message string, no surrounding markdown or explanations."
    PROMPT="$FUNCTION_PROMPT

Context:
repositoryPath: $REPO_PATH
currentMessage:
\`\`\`
$CURRENT_MESSAGE
\`\`\`

User prompt:
$USER_PROMPT"

    # Report progress
    echo "[progress] Generating commit message..." >&2

    if [ -n "$REPO_PATH" ]; then
        cd "$REPO_PATH"
    fi

    # Call Gemini - it should only output the commit message
    # Use --yolo mode to auto-approve all actions and avoid any interactive prompts
    COMMIT_MESSAGE=$("$GEMINI_CMD" --yolo "$PROMPT" 2>/dev/null)
fi

# Check if we got a response
if [ -z "$COMMIT_MESSAGE" ]; then
    echo "[error] Failed to generate commit message" >&2
    # If there was output but we couldn't extract a message, show what we got
    if [ -s "$TEMP_OUTPUT" ]; then
        echo "[error] Gemini output:" >&2
        head -20 "$TEMP_OUTPUT" | while IFS= read -r line; do
            echo "[error]   $line" >&2
        done
    fi
    exit 1
fi

# Output the commit message to stdout
echo "$COMMIT_MESSAGE"

# Explicitly exit successfully to ensure the script terminates
exit 0
