// The Cocoa GCCommitDetailsViewController is available as a JavaScript variable:
// window.CocoaController
// logging can be done with: CocoaController.log(CocoaController.displayedCommit.subject());

// all loaded commit html strings are first added to a buffer and output when appropriate
var htmlBuffer = "";
// timestamp of the last time new commits were loaded
var loadingTriggeredDate = false;
// how many new items will be loaded with the next loading action
//var loadMoreItemsAmount = 0;
// how many items are currently shown
var currentItemCount = 0;

var constants = {
	changedFilesContainerCollapsedHeight: 105,
	oneLineOfRefsHeight: 18,
	minimumDisplaySecondsForLoadingMessage: (1*1000),
	minimumSecondsBetweenLoadings: (1*1000),
	gravatarPlaceholderImagePath: '../img/avatarDefaultPlaceholder@2x.png'
};



/**
 Replaces the list of commits with the given HTML string of commits
 */
function insertCommitsIntoDOM(htmlString, newCurrentItemCount, expandChangedFiles){
	// if we already have those items, don't show them again (actually shouldn't happen - just for error protection!)
	if(newCurrentItemCount <= currentItemCount){
		return;	
	}
	
	// add new content to buffer
	htmlBuffer += htmlString;
	currentItemCount = newCurrentItemCount;
	
	// check if buffer content shall already be output or not (is the case when the loading wheel shall be shown a little longer...)
	if(loadingTriggeredDate && 
	   (new Date() - loadingTriggeredDate < constants.minimumDisplaySecondsForLoadingMessage)){
		setTimeout(function(){
				   flushHtmlBuffer(expandChangedFiles);
				   }, ((loadingTriggeredDate - new Date()) + constants.minimumDisplaySecondsForLoadingMessage + 100));
	}else{
		// show the content directly
		flushHtmlBuffer(expandChangedFiles);
	}
}


/**
 Outputs all new content from the htmlBuffer
 Using a buffer ensures that all new content is added in the correct order 
 (NOTE: since we're delaying the insertion of new content until the loader wheel has been shown for long enough, 
 insertion order can be a problem)
 */
function flushHtmlBuffer(expandChangedFiles){
	var buffer = $(htmlBuffer);
	// show additional commits and hide loadingMessage
	$("#loadingMessage").hide();
	$("#noCommitsMessageArea").hide();
	setupOptionsMenu(buffer);
	setupContextualMenu(buffer);
    setupChangedFileMenus(buffer);
	$("#contentArea").append(buffer);
	toggleFileDisplay(expandChangedFiles);
	$("#contentArea").show();
	$(".subject").ellipsis(true, true, "", "");
	
	// clear buffer
	htmlBuffer = "";
}


/**
 Update the actor/committer image with a new one that was retrieved from remote
 */
function updateLoadedImage(actorEmailClass, imageTag){
	var imageElements = document.getElementsByClassName(actorEmailClass);	// jQuery doesn't seem to like Classes with characters like .- 

	if(imageElements){
		for(var i=0; i < imageElements.length; i++){
			$(imageElements[i]).replaceWith(imageTag);
		}
	}
}


/**
 Shows or hides _all_ changedFiles containers
 */
function toggleFileDisplay(show) {
	if(show == 1){
		$('.changedFilesWrapper').show();
		$('.toggleExpansionButton').addClass('expanded').removeClass('collapsed');
	}else{
		$('.changedFilesWrapper').hide();
		$('.toggleExpansionButton').addClass('collapsed').removeClass('expanded');
	}
}


/**
 Triggers the opening of a HUD window, showing the diff for a single file
 */
function showFileDiffInOverlay(commitHash, filePath){
	CocoaController.openNewWindowWithFileDiff(filePath, commitHash);
}


/**
 Expands or collapses the changedFilesContainer, depending on its current state
 */
function toggleMoreLines(clickedMoreButton, event){
	// which 'changedFilesContainer' is concerned?
	var changedFilesContainer = clickedMoreButton.closest('.commitItem').find('.changedFilesWrapper');
	changedFilesContainer.toggle();
	if(clickedMoreButton.hasClass('collapsed')){
		clickedMoreButton.removeClass('collapsed');
		clickedMoreButton.addClass('expanded');
	}else{
		clickedMoreButton.removeClass('expanded');
		clickedMoreButton.addClass('collapsed');
	}
}


function expandChangedFilesContainer(clickedMoreButton, changedFilesContainer){
	clickedMoreButton.addClass('expanded');
	changedFilesContainer.addClass('expanded');
	changedFilesContainer.parent().addClass('expanded');
}


function collapseChangedFilesContainer(clickedMoreButton, changedFilesContainer){
	clickedMoreButton.removeClass('expanded');
	changedFilesContainer.removeClass('expanded');
	changedFilesContainer.parent().removeClass('expanded');
	// check if container is still in the visible scrolling area
	var parentCommitContainer = changedFilesContainer.closest('.commitItem');
	if($(window).scrollTop() > parentCommitContainer.offset().top){
		$(window).scrollTop(parentCommitContainer.offset().top);
	}
}


/**
 Enables the "option gear" menu for all elements in a certain DOM fragment
 */
function setupOptionsMenu(domFragment){
	$(".optionsMenuButton", domFragment).contextMenu('options-menu', 
												optionsMenuOptions,
												{	callback: optionsMenuItemClicked,
													disable_native_context_menu: true, 
													activateOnLeftClick:true,
													align: 'right' }
												);	
}

/**
 Enables the contextual menu for all elements in a certain DOM fragment
 */
function setupContextualMenu(domFragment){
	// options for the contextual menu are passed as JSON string - eval it first
	$(".basicInfoContainer", domFragment).contextMenu('options-menu', 
													  optionsMenuOptions,
													  {	callback: optionsMenuItemClicked,
														disable_native_context_menu: true, 
														activateOnLeftClick:false,
														align: 'left' }
													  );
    // same menu - but if user clicks on .hashesContainer, open the menu aligned _right_
	$(".hashesContainer", domFragment).contextMenu('options-menu', 
													  optionsMenuOptions,
													  {	callback: optionsMenuItemClicked,
                                                      disable_native_context_menu: true, 
                                                      activateOnLeftClick:false,
                                                      align: 'right' }
													  );	
}


/**
 Adds contextual menus to all relevant changedFilesLines
 */
function setupChangedFileMenus(domFragment){
	$(".changedFilesLine", domFragment).each(function(index, el){
								// don't allow contextual menu on deleted file
								if($('.deleted',el).length > 0) return;
								
								$(el).contextMenu('context-menu-committed-file', 
												  [ { title: 'Restore File in Working Directory...',
                                                      cocoaAction: 'restoreFileInWorkingDirectoryMenuItemOptionClicked:forCommit:'},
                                                    { title: 'Show File History',
                                                      cocoaAction: 'showFileHistoryMenuItemOptionClicked:forCommit:'}
                                                   ],
												  { callback: changedFileMenuItemClicked,
                                                    disable_native_context_menu: true, 
                                                    activateOnLeftClick:false,
                                                    align: 'left' }
												  );
								});
}


function setOptionsMenuOptions(jsonOptions){
	eval('optionsMenuOptions = ' + jsonOptions);
}


function commitHashForDomElement(domElement){
	var commitItemRootEl = $(domElement).closest('.commitItem');
	if(commitItemRootEl){
		return commitItemRootEl.attr('commitHash');
	}else{
		return false;
	}
}


function optionsMenuItemClicked(menuOption, clickedItem){
	var commitHash = commitHashForDomElement(clickedItem);
	// call cocoa controller
	CocoaController.contextualMenuItemClicked(menuOption.cocoaAction, commitHash);
}


function changedFileMenuItemClicked(menuOption, clickedItem){
	var commitHash = commitHashForDomElement(clickedItem);
	if(menuOption.cocoaAction == 'restoreFileInWorkingDirectoryMenuItemOptionClicked:forCommit:' ||
       menuOption.cocoaAction == 'showFileHistoryMenuItemOptionClicked:forCommit:'){
		var itemPath = $('.changedFilePathContent', clickedItem).attr('filePath');
	}else{
		var itemPath = false;
	}
	
	// call cocoa controller
	CocoaController.changedFileMenuItemClicked(menuOption.cocoaAction, itemPath, commitHash);
}


function hideLoadingMessageArea(){
	$('#loadingMessage').hide();
}


function toggleNoCommitsMessageArea(show){
	if(show){
		$('#noCommitsMessageArea').show();
	}else{
		$('#noCommitsMessageArea').hide();
	}
}


/**
 Upon reaching the page bottom, new items should be loaded automatically
 */
function pageBottomReached(){
	if(loadMoreItemsAmount > 0 && 
	   (!loadingTriggeredDate || (loadingTriggeredDate && (new Date() - loadingTriggeredDate > constants.minimumSecondsBetweenLoadings)) ) ){
		$("#loadingMessage").html("Loading " + loadMoreItemsAmount + " more commits...").show();
		loadingTriggeredDate = new Date();
		CocoaController.loadMoreCommits();
	}
}




/**
 Bind event handlers to existing (and future!!) items
 */
function bindEventHandlers(){
	// double click on a commit item
	// NOTE: didn't feel right to open details on dblclick, so we disable it here...
	/*	$('.commitItem').live('dblclick', function() {
	 var commitHash = commitHashForDomElement($(this));
	 CocoaController.commitItemDoubleClicked(commitHash);
	 }); */
	$(window).scroll(function(){
						if( $(window).scrollTop() == ($(document).height() - $(window).height()) ){
							pageBottomReached();
						}
					 });
	
	// toggle more changed lines button
	$('.toggleExpansionButton').live('click', function(event) {
									 toggleMoreLines($(this), event);
									 return false;
									 });
}




/**
 As soon as page is loaded, starts to setup content
 */
$(function(){
  //  CocoaController.log("Hello to Cocoa NSLog from JavaScript");
  bindEventHandlers();
});
