//
//  FNOperation.h
//  FNFoundation
//
//  Created by Alexander Rinass on 24/10/15.
//  Copyright © 2015 Fournova GmbH. All rights reserved.
//

#import <Foundation/Foundation.h>

@interface FNOperation : NSOperation
@property (atomic) NSError *error;

/**
 Signals that the operation is actually beginning its work.

 This method is always called.
 */
- (void)operationDidStart;

/**
 Called directly before `execute`. 
 
 This method can be used to group common actions in parent classes, e.g. 
 validate the operation for execution. To prevent the operation from further 
 execution, call `finish` or `finishWithError:`.
 
 If the operation has been cancelled or finished through `finish` or
 `finishWithError:`, this method is not executed.
 */
- (void)operationWillExecute;

/**
 The main execution method of the operation.

 If the operation has been cancelled or finished through `finish` or 
 `finishWithError:`, this method is not executed.
 */
- (void)execute;

/**
 Called directly after `execute`.
 
 This method can be used to group common post-execution actions in parent
 classes, like processing a result.

 If the operation has been cancelled or finished through `finish` or
 `finishWithError:`, this method is not executed.
 */
- (void)operationDidExecute;

/**
 Signals that the operation is finished. The state of `isFinished` is `YES` when
 the method is called.

 This method is always called.
 */
- (void)operationDidFinish;

/**
 Finishes the operation and prevents any further execution.
 */
- (void)finish;

/**
 Convenience method to set an error and finish the operation.
 */
- (void)finishWithError:(NSError *)error;

@end
